<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class ECF7L_Leads {

    public static function table() {
        global $wpdb;
        return $wpdb->prefix . 'ecf7l_leads';
    }

    public static function insert( $data ) {
        global $wpdb;

        $defaults = array(
            'form_id'     => 0,
            'form_title'  => '',
            'name'        => '',
            'email'       => '',
            'phone'       => '',
            'status'      => 'nuevo',
            'tags'        => '',
            'source'      => '',
            'assigned_to' => 0,
            'ip_address'  => '',
            'user_agent'  => '',
            'data'        => array(),
            'notes'       => '',
        );

        $data = wp_parse_args( $data, $defaults );

        $wpdb->insert(
            self::table(),
            array(
                'form_id'      => (int) $data['form_id'],
                'form_title'   => $data['form_title'],
                'submitted_at' => current_time( 'mysql' ),
                'name'         => $data['name'],
                'email'        => $data['email'],
                'phone'        => $data['phone'],
                'status'       => $data['status'],
                'tags'         => is_array( $data['tags'] ) ? implode( ',', $data['tags'] ) : $data['tags'],
                'source'       => $data['source'],
                'assigned_to'  => (int) $data['assigned_to'],
                'ip_address'   => $data['ip_address'],
                'user_agent'   => $data['user_agent'],
                'data'         => maybe_serialize( $data['data'] ),
                'notes'        => $data['notes'],
            )
        );

        return $wpdb->insert_id;
    }

    public static function update( $id, $fields ) {
        global $wpdb;

        $allowed = array(
            'name',
            'email',
            'phone',
            'status',
            'tags',
            'source',
            'assigned_to',
            'notes',
        );

        $data = array();
        foreach ( $allowed as $key ) {
            if ( isset( $fields[ $key ] ) ) {
                if ( 'assigned_to' === $key ) {
                    $data[ $key ] = (int) $fields[ $key ];
                } elseif ( 'tags' === $key && is_array( $fields[ $key ] ) ) {
                    $data[ $key ] = implode( ',', $fields[ $key ] );
                } else {
                    $data[ $key ] = $fields[ $key ];
                }
            }
        }

        if ( empty( $data ) ) {
            return false;
        }

        return $wpdb->update(
            self::table(),
            $data,
            array( 'id' => (int) $id ),
            null,
            array( '%d' )
        );
    }

    public static function delete( $id ) {
        global $wpdb;

        return $wpdb->delete(
            self::table(),
            array( 'id' => (int) $id ),
            array( '%d' )
        );
    }

    public static function get( $id ) {
        global $wpdb;

        return $wpdb->get_row(
            $wpdb->prepare(
                'SELECT * FROM ' . self::table() . ' WHERE id = %d',
                $id
            ),
            ARRAY_A
        );
    }

    /**
     * Listado de leads con filtros, ordenación y paginación.
     */
    public static function get_all( $args = array() ) {
        global $wpdb;

        $defaults = array(
            'paged'           => 1,
            'per_page'        => 20,
            'search'          => '',
            'status'          => '',
            'form_id'         => '',
            'date_from'       => '',
            'date_to'         => '',
            'assigned_to'     => '',   // '', '0' (sin asignar) o ID de usuario
            'unassigned_only' => 0,    // 1 = solo sin asignar
            'orderby'         => 'date', // id, form, name, email, phone, status, date
            'order'           => 'desc',
        );
        $args = wp_parse_args( $args, $defaults );

        $where  = array( '1=1' );
        $params = array();

        // Búsqueda
        if ( $args['search'] ) {
            $like = '%' . $wpdb->esc_like( $args['search'] ) . '%';
            $where[]  = '(name LIKE %s OR email LIKE %s OR phone LIKE %s)';
            $params[] = $like;
            $params[] = $like;
            $params[] = $like;
        }

        // Estado
        if ( $args['status'] ) {
            $where[]  = 'status = %s';
            $params[] = $args['status'];
        }

        // Formulario
        if ( $args['form_id'] ) {
            $where[]  = 'form_id = %d';
            $params[] = (int) $args['form_id'];
        }

        // Fechas
        if ( $args['date_from'] ) {
            $where[]  = 'submitted_at >= %s';
            $params[] = $args['date_from'] . ' 00:00:00';
        }

        if ( $args['date_to'] ) {
            $where[]  = 'submitted_at <= %s';
            $params[] = $args['date_to'] . ' 23:59:59';
        }

        // Responsable / sin asignar
        $assigned_to     = $args['assigned_to'];
        $unassigned_only = ! empty( $args['unassigned_only'] );

        if ( $unassigned_only ) {
            $where[] = 'assigned_to = 0';
        } elseif ( '' !== $assigned_to ) {
            $assigned_int = (int) $assigned_to;
            if ( $assigned_int > 0 ) {
                $where[]  = 'assigned_to = %d';
                $params[] = $assigned_int;
            } else {
                // '0' = sin asignar
                $where[] = 'assigned_to = 0';
            }
        }

        $where_sql = implode( ' AND ', $where );
        $table     = self::table();

        $paged    = max( 1, (int) $args['paged'] );
        $per_page = max( 1, (int) $args['per_page'] );
        $offset   = ( $paged - 1 ) * $per_page;

        // Ordenación
        $orderby_key = isset( $args['orderby'] ) ? $args['orderby'] : 'date';
        $order       = isset( $args['order'] ) ? strtolower( $args['order'] ) : 'desc';
        $order       = ( 'asc' === $order ) ? 'ASC' : 'DESC';

        $allowed_orderby = array(
            'id'     => 'id',
            'form'   => 'form_title',
            'name'   => 'name',
            'email'  => 'email',
            'phone'  => 'phone',
            'status' => 'status',
            'date'   => 'submitted_at',
        );

        $orderby_sql = isset( $allowed_orderby[ $orderby_key ] ) ? $allowed_orderby[ $orderby_key ] : 'submitted_at';

        $sql_items = "SELECT * FROM $table WHERE $where_sql ORDER BY $orderby_sql $order LIMIT %d OFFSET %d";
        $sql_count = "SELECT COUNT(*) FROM $table WHERE $where_sql";

        // items
        $items_params   = $params;
        $items_params[] = $per_page;
        $items_params[] = $offset;

        $items = $wpdb->get_results(
            $wpdb->prepare( $sql_items, $items_params ),
            ARRAY_A
        );

        // total (evitar notice de prepare sin placeholders)
        if ( ! empty( $params ) ) {
            $total = (int) $wpdb->get_var(
                $wpdb->prepare( $sql_count, $params )
            );
        } else {
            $total = (int) $wpdb->get_var( $sql_count );
        }

        return array(
            'items' => $items,
            'total' => $total,
        );
    }

    public static function get_statuses() {
        return array(
            'nuevo'       => __( 'Nuevo', 'easycf7-leads' ),
            'contactado'  => __( 'Contactado', 'easycf7-leads' ),
            'en_proceso'  => __( 'En proceso', 'easycf7-leads' ),
            'ganado'      => __( 'Ganado', 'easycf7-leads' ),
            'perdido'     => __( 'Perdido', 'easycf7-leads' ),
        );
    }

    /**
     * Resumen general (total, últimos 7 días, estados, top formularios).
     * Opcionalmente filtrado por ID de formulario.
     */
    public static function get_stats_overview( $form_id = 0 ) {
        global $wpdb;
        $table = self::table();

        $where  = '1=1';
        $params = array();

        if ( $form_id ) {
            $where   .= ' AND form_id = %d';
            $params[] = (int) $form_id;
        }

        // Total leads
        if ( ! empty( $params ) ) {
            $total = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE $where",
                    $params
                )
            );
        } else {
            $total = (int) $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE $where" );
        }

        // Últimos 7 días
        $from_7 = gmdate(
            'Y-m-d H:i:s',
            current_time( 'timestamp', true ) - ( 7 * DAY_IN_SECONDS )
        );

        $where_last  = $where . ' AND submitted_at >= %s';
        $params_last = $params;
        $params_last[] = $from_7;

        $last7 = (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE $where_last",
                $params_last
            )
        );

        // Por estado
        $statuses  = self::get_statuses();
        $by_status = array();

        foreach ( $statuses as $key => $label ) {
            $status_where  = 'status = %s';
            $status_params = array( $key );

            if ( $form_id ) {
                $status_where   .= ' AND form_id = %d';
                $status_params[] = (int) $form_id;
            }

            $count = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE $status_where",
                    $status_params
                )
            );

            $by_status[ $key ] = array(
                'label' => $label,
                'count' => $count,
            );
        }

        // Top formularios (globales o filtrados)
        if ( $form_id ) {
            $top_forms_params = array( (int) $form_id );
            $top_forms_sql    = "SELECT form_title, COUNT(*) as total FROM $table WHERE form_id = %d GROUP BY form_id, form_title ORDER BY total DESC LIMIT 5";

            $top_forms = $wpdb->get_results(
                $wpdb->prepare( $top_forms_sql, $top_forms_params ),
                ARRAY_A
            );
        } else {
            $top_forms_sql = "SELECT form_title, COUNT(*) as total FROM $table GROUP BY form_id, form_title ORDER BY total DESC LIMIT 5";
            $top_forms     = $wpdb->get_results( $top_forms_sql, ARRAY_A );
        }

        return array(
            'total'      => $total,
            'last7'      => $last7,
            'by_status'  => $by_status,
            'top_forms'  => $top_forms,
        );
    }

    /**
     * Leads por día (últimos X días), opcionalmente por formulario.
     */
    public static function get_leads_by_day( $days = 7, $form_id = 0 ) {
        global $wpdb;

        $days  = max( 1, (int) $days );
        $table = self::table();

        $now_ts  = current_time( 'timestamp', true );
        $from_ts = $now_ts - ( $days * DAY_IN_SECONDS );

        $from_mysql = gmdate( 'Y-m-d H:i:s', $from_ts );

        $where  = 'submitted_at >= %s';
        $params = array( $from_mysql );

        if ( $form_id ) {
            $where   .= ' AND form_id = %d';
            $params[] = (int) $form_id;
        }

        $sql = "SELECT DATE(submitted_at) AS day, COUNT(*) AS total
                FROM $table
                WHERE $where
                GROUP BY DATE(submitted_at)
                ORDER BY day ASC";

        $rows = $wpdb->get_results(
            $wpdb->prepare( $sql, $params ),
            ARRAY_A
        );

        $map = array();
        if ( $rows ) {
            foreach ( $rows as $row ) {
                $map[ $row['day'] ] = (int) $row['total'];
            }
        }

        $data = array();
        for ( $i = $days - 1; $i >= 0; $i-- ) {
            $ts   = $now_ts - ( $i * DAY_IN_SECONDS );
            $date = gmdate( 'Y-m-d', $ts );

            $data[] = array(
                'date'  => $date,
                'total' => isset( $map[ $date ] ) ? $map[ $date ] : 0,
            );
        }

        return $data;
    }

    /**
     * KPIs de conversión: ganados, total, ratio y tiempo medio (en días) hasta lead ganado.
     */
    public static function get_conversion_kpis( $form_id = 0 ) {
        global $wpdb;
        $table = self::table();

        $where  = '1=1';
        $params = array();

        if ( $form_id ) {
            $where   .= ' AND form_id = %d';
            $params[] = (int) $form_id;
        }

        // Total leads
        if ( ! empty( $params ) ) {
            $total = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE $where",
                    $params
                )
            );
        } else {
            $total = (int) $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE $where" );
        }

        // Ganados
        $where_won  = $where . " AND status = 'ganado'";
        $params_won = $params;

        if ( ! empty( $params_won ) ) {
            $won = (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE $where_won",
                    $params_won
                )
            );
        } else {
            $won = (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM $table WHERE $where_won"
            );
        }

        $win_rate = 0;
        if ( $total > 0 && $won > 0 ) {
            $win_rate = ( $won / $total ) * 100;
        }

        // Tiempo medio hasta ganado (NOW - submitted_at) en días
        $avg_days_to_win = null;

        if ( $won > 0 ) {
            if ( ! empty( $params_won ) ) {
                $avg_days_to_win = (float) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT AVG(DATEDIFF(NOW(), submitted_at)) FROM $table WHERE $where_won",
                        $params_won
                    )
                );
            } else {
                $avg_days_to_win = (float) $wpdb->get_var(
                    "SELECT AVG(DATEDIFF(NOW(), submitted_at)) FROM $table WHERE $where_won"
                );
            }
        }

        return array(
            'total'           => $total,
            'won'             => $won,
            'win_rate'        => $win_rate,
            'avg_days_to_win' => $avg_days_to_win,
        );
    }
}
