<?php
/*
Plugin Name: EasyCF7 Leads
Description: CRM ligero para Contact Form 7: guarda leads, estados, etiquetas, WhatsApp redirect, dashboard, exportación CSV y webhooks.
Version: 1.0.0
Author: Huwawa
Text Domain: easycf7-leads
*/

if ( ! function_exists( 'elp_fs' ) ) {
    function elp_fs() {
        global $elp_fs;

        if ( ! isset( $elp_fs ) ) {
            $sdk_path = __DIR__ . '/vendor/freemius/start.php';

            if ( file_exists( $sdk_path ) ) {
                require_once $sdk_path;

                $elp_fs = fs_dynamic_init( array(
                    'id'             => '21994',
                    'slug'           => 'easycf7-leads-pro',
                    'type'           => 'plugin',
                    'public_key'     => 'pk_2deb90c1a605dbfad4c47ea193949',
                    'is_premium'     => true,
                    'has_addons'     => false,
                    'has_paid_plans' => true,
                    'menu'           => array(
                        'slug'    => 'easycf7-leads-dashboard',
                        'support' => false,
                        'pricing' => false,
                        'account' => true,
                    ),
                ) );
            } else {
                return null;
            }
        }

        return $elp_fs;
    }

    // Init Freemius.
    if ( elp_fs() ) {
        do_action( 'elp_fs_loaded' );
    }
}

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'ECF7L_VERSION', '3.0.0' );
define( 'ECF7L_FILE', __FILE__ );
define( 'ECF7L_DIR', plugin_dir_path( __FILE__ ) );
define( 'ECF7L_URL', plugin_dir_url( __FILE__ ) );

require_once ECF7L_DIR . 'includes/class-ecf7l-leads.php';
require_once ECF7L_DIR . 'includes/class-ecf7l-webhooks.php';
require_once ECF7L_DIR . 'includes/class-ecf7l-exporter.php';
require_once ECF7L_DIR . 'includes/class-ecf7l-notes.php';
require_once ECF7L_DIR . 'includes/class-ecf7l-geoip.php';

function ecf7l_is_pro() {
    if ( function_exists( 'elp_fs' ) ) {
        $fs = elp_fs();
        if ( $fs && method_exists( $fs, 'can_use_premium_code' ) && $fs->can_use_premium_code() ) {
            return true;
        }
    }
    return false;
}

function ecf7l_activate() {
    global $wpdb;

    $table_leads = ECF7L_Leads::table();
    $charset_collate = $wpdb->get_charset_collate();

    $sql_leads = "CREATE TABLE $table_leads (
      id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
      form_id BIGINT UNSIGNED NOT NULL,
      form_title VARCHAR(255) NOT NULL,
      submitted_at DATETIME NOT NULL,
      name VARCHAR(190) NULL,
      email VARCHAR(190) NULL,
      phone VARCHAR(190) NULL,
      status VARCHAR(50) NULL,
      tags TEXT NULL,
      source VARCHAR(190) NULL,
      assigned_to BIGINT UNSIGNED NULL,
      ip_address VARCHAR(100) NULL,
      user_agent TEXT NULL,
      data LONGTEXT NOT NULL,
      notes TEXT NULL,
      PRIMARY KEY  (id),
      KEY form_id (form_id),
      KEY submitted_at (submitted_at),
      KEY status (status)
    ) $charset_collate;";

    $table_notes = $wpdb->prefix . 'ecf7l_lead_notes';

    $sql_notes = "CREATE TABLE $table_notes (
      id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
      lead_id BIGINT UNSIGNED NOT NULL,
      type VARCHAR(50) NOT NULL DEFAULT 'note',
      message TEXT NOT NULL,
      created_by BIGINT UNSIGNED NULL,
      created_at DATETIME NOT NULL,
      PRIMARY KEY  (id),
      KEY lead_id (lead_id),
      KEY created_at (created_at)
    ) $charset_collate;";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql_leads );
    dbDelta( $sql_notes );

    if ( ! get_option( 'ecf7l_settings' ) ) {
        add_option( 'ecf7l_settings', array(
            'enable_logging'          => 1,
            'notify_assignee_email'   => 0,
            'whatsapp_enabled'        => 1,
            'whatsapp_default_number' => '',
            'whatsapp_forms'          => array(),
            'webhooks'                => array(),
        ) );
    }

    if ( ! get_option( 'ecf7l_license_key' ) ) {
        add_option( 'ecf7l_license_key', '' );
    }

    if ( ! get_option( 'ecf7l_license_status' ) ) {
        add_option( 'ecf7l_license_status', 'free' );
    }
}
register_activation_hook( __FILE__, 'ecf7l_activate' );

/**
 * Dependencias.
 */
function ecf7l_check_dependencies() {
    include_once ABSPATH . 'wp-admin/includes/plugin.php';
    if ( ! is_plugin_active( 'contact-form-7/wp-contact-form-7.php' ) ) {
        add_action( 'admin_notices', function () {
            echo '<div class="notice notice-error"><p><strong>EasyCF7 Leads PRO</strong> requiere <strong>Contact Form 7</strong> activo.</p></div>';
        } );
    }
}
add_action( 'admin_init', 'ecf7l_check_dependencies' );

/**
 * Assets.
 */
function ecf7l_enqueue_admin_assets( $hook ) {
    if ( strpos( $hook, 'easycf7-leads' ) === false ) {
        return;
    }

    wp_enqueue_style(
        'ecf7l-admin',
        ECF7L_URL . 'assets/css/admin.css',
        array(),
        ECF7L_VERSION
    );

    wp_enqueue_script(
        'ecf7l-admin',
        ECF7L_URL . 'assets/js/admin.js',
        array( 'jquery' ),
        ECF7L_VERSION,
        true
    );

    wp_localize_script( 'ecf7l-admin', 'ECF7L_Admin', array(
        'is_pro' => ecf7l_is_pro() ? 1 : 0,
    ) );
}
add_action( 'admin_enqueue_scripts', 'ecf7l_enqueue_admin_assets' );

function ecf7l_enqueue_frontend_assets() {
    wp_enqueue_style(
        'ecf7l-frontend',
        ECF7L_URL . 'assets/css/frontend.css',
        array(),
        ECF7L_VERSION
    );

    wp_enqueue_script(
        'ecf7l-frontend',
        ECF7L_URL . 'assets/js/frontend.js',
        array(),
        ECF7L_VERSION,
        true
    );

    $settings       = get_option( 'ecf7l_settings', array() );
    $whatsapp_forms = isset( $settings['whatsapp_forms'] ) ? $settings['whatsapp_forms'] : array();
    $enabled        = ! empty( $settings['whatsapp_enabled'] );

    wp_localize_script( 'ecf7l-frontend', 'ECF7L_WhatsApp', array(
        'enabled' => $enabled ? 1 : 0,
        'forms'   => $whatsapp_forms,
    ) );
}
add_action( 'wp_enqueue_scripts', 'ecf7l_enqueue_frontend_assets' );

/**
 * Captura de leads desde CF7.
 */
/**
 * Captura de leads desde CF7.
 *
 * Soporta tanto wpcf7_mail_sent con 1 parámetro (CF7 actual)
 * como posibles versiones con 2 parámetros en el futuro.
 */
function ecf7l_cf7_store_lead( $contact_form, $result = null ) {
    // Seguridad extra: si no viene un objeto válido, salimos.
    if ( ! $contact_form || ! is_object( $contact_form ) ) {
        return;
    }

    $settings = get_option( 'ecf7l_settings', array() );
    if ( empty( $settings['enable_logging'] ) ) {
        return;
    }

    if ( ! class_exists( 'WPCF7_Submission' ) ) {
        return;
    }

    $submission = WPCF7_Submission::get_instance();
    if ( ! $submission ) {
        return;
    }

    $posted_data = $submission->get_posted_data();
    if ( ! is_array( $posted_data ) ) {
        $posted_data = array();
    }

    if ( ! method_exists( $contact_form, 'id' ) ) {
        return;
    }

    $form_id    = $contact_form->id();
    $form_title = method_exists( $contact_form, 'title' ) ? $contact_form->title() : '';

    // Campos típicos
    $name  = isset( $posted_data['your-name'] ) ? sanitize_text_field( $posted_data['your-name'] ) : '';
    $email = isset( $posted_data['your-email'] ) ? sanitize_email( $posted_data['your-email'] ) : '';
    $phone = isset( $posted_data['your-phone'] ) ? sanitize_text_field( $posted_data['your-phone'] ) : '';

    $ip = $submission->get_meta( 'remote_ip' );
    $ua = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

    $lead_id = ECF7L_Leads::insert( array(
        'form_id'     => $form_id,
        'form_title'  => $form_title,
        'name'        => $name,
        'email'       => $email,
        'phone'       => $phone,
        'data'        => $posted_data,
        'ip_address'  => $ip,
        'user_agent'  => $ua,
        'source'      => isset( $_SERVER['HTTP_REFERER'] ) ? esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) : '',
    ) );
    
    if ( $lead_id && class_exists( 'ECF7L_Notes' ) ) {
        ECF7L_Notes::log_event(
            $lead_id,
            sprintf(
                /* translators: %s: titulo del formulario */
                __( 'Lead creado automáticamente desde el formulario "%s".', 'easycf7-leads' ),
                $form_title
            )
        );
    }
    

    if ( function_exists( 'ecf7l_is_pro' ) && ecf7l_is_pro() && $lead_id && class_exists( 'ECF7L_Webhooks' ) ) {
        ECF7L_Webhooks::send_lead( $lead_id );
    }
    
}
add_action( 'wpcf7_mail_sent', 'ecf7l_cf7_store_lead', 10, 2 );




/**
 * Menú de administración.
 */
function ecf7l_admin_menu() {
    add_menu_page(
        __( 'EasyCF7 Leads', 'easycf7-leads' ),
        'EasyCF7 Leads',
        'manage_options',
        'easycf7-leads-dashboard',
        'ecf7l_render_dashboard_page',
        'dashicons-chart-pie',
        26
    );

    add_submenu_page(
        'easycf7-leads-dashboard',
        __( 'Upgrade a PRO', 'easycf7-leads' ),
        __( 'Upgrade a PRO', 'easycf7-leads' ),
        __( 'Dashboard', 'easycf7-leads' ),
        __( 'Dashboard', 'easycf7-leads' ),
        'manage_options',
        'easycf7-leads-dashboard',
        'ecf7l_render_dashboard_page'
    );

    add_submenu_page(
        'easycf7-leads-dashboard',
        __( 'Leads', 'easycf7-leads' ),
        __( 'Leads', 'easycf7-leads' ),
        'manage_options',
        'easycf7-leads',
        'ecf7l_render_leads_page'
    );

    add_submenu_page(
        'easycf7-leads-dashboard',
        __( 'Ajustes', 'easycf7-leads' ),
        __( 'Ajustes', 'easycf7-leads' ),
        'manage_options',
        'easycf7-leads-settings',
        'ecf7l_render_settings_page'
    );
}
add_action( 'admin_menu', 'ecf7l_admin_menu' );

/**
 * Dashboard.
 */
function ecf7l_render_dashboard_page() {
    include ECF7L_DIR . 'views/admin-dashboard.php';
}

/**
 * Página de leads (lista o detalle).
 */
function ecf7l_render_leads_page() {
    if ( isset( $_GET['view'] ) && $_GET['view'] === 'detail' && ! empty( $_GET['lead_id'] ) ) {
        include ECF7L_DIR . 'views/admin-lead-detail.php';
    } else {
        include ECF7L_DIR . 'views/admin-leads-list.php';
    }
}

/**
 * Página de ajustes.
 */
function ecf7l_render_settings_page() {
    include ECF7L_DIR . 'views/admin-settings.php';
}


function ecf7l_render_upgrade_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $message = '';

    if ( isset( $_POST['ecf7l_license_key'] ) && check_admin_referer( 'ecf7l_save_license', 'ecf7l_license_nonce' ) ) {
        $key = isset( $_POST['ecf7l_license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['ecf7l_license_key'] ) ) : '';
        update_option( 'ecf7l_license_key', $key );

        if ( $key !== '' ) {
            update_option( 'ecf7l_license_status', 'valid' );
            $message = __( 'Licencia activada correctamente.', 'easycf7-leads' );
        } else {
            update_option( 'ecf7l_license_status', 'free' );
            $message = __( 'Licencia desactivada.', 'easycf7-leads' );
        }
    }

    $license_key    = get_option( 'ecf7l_license_key', '' );
    $license_status = get_option( 'ecf7l_license_status', 'free' );
    ?>
    <div class="wrap ecf7l-wrap">
        <div class="ecf7l-header">
            <div class="ecf7l-header-main">
                <div class="ecf7l-logo-badge">EL</div>
                <div>
                    <h1 class="ecf7l-header-title">Upgrade a EasyCF7 Leads PRO</h1>
                    <p class="ecf7l-header-subtitle">
                        Desbloquea funciones avanzadas: Mis leads, asignación, GeoIP, webhooks y más.
                    </p>
                </div>
            </div>
            <div class="ecf7l-header-side">
                <div class="ecf7l-badge-pill">
                    <span class="dot"></span>
                    <span><?php echo ecf7l_is_pro() ? 'Licencia PRO activa' : 'Modo FREE'; ?></span>
                </div>
            </div>
        </div>

        <?php if ( $message ) : ?>
            <div class="notice notice-success is-dismissible">
                <p><?php echo esc_html( $message ); ?></p>
            </div>
        <?php endif; ?>

        <h2><?php esc_html_e( 'Introduce tu clave de licencia', 'easycf7-leads' ); ?></h2>

        <form method="post">
            <?php wp_nonce_field( 'ecf7l_save_license', 'ecf7l_license_nonce' ); ?>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="ecf7l_license_key"><?php esc_html_e( 'Clave de licencia', 'easycf7-leads' ); ?></label>
                    </th>
                    <td>
                        <input
                            name="ecf7l_license_key"
                            id="ecf7l_license_key"
                            type="text"
                            class="regular-text"
                            value="<?php echo esc_attr( $license_key ); ?>"
                            placeholder="EASYCF7-PRO-XXXX"
                        />
                        <p class="description">
                            <?php esc_html_e( 'Copia y pega aquí la clave de licencia que has recibido al comprar EasyCF7 Leads PRO.', 'easycf7-leads' ); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'Estado de la licencia', 'easycf7-leads' ); ?></th>
                    <td>
                        <?php
                        if ( $license_status === 'valid' && $license_key !== '' ) {
                            echo '<span class="ecf7l-license-status-valid">' . esc_html__( 'Licencia PRO activa', 'easycf7-leads' ) . '</span>';
                        } else {
                            echo '<span class="ecf7l-license-status-free">' . esc_html__( 'Modo FREE', 'easycf7-leads' ) . '</span>';
                        }
                        ?>
                    </td>
                </tr>
            </table>
            <?php submit_button( __( 'Guardar licencia', 'easycf7-leads' ) ); ?>
        </form>

        <div class="ecf7l-box-pro">
            <p><strong><?php esc_html_e( '¿Qué incluye EasyCF7 Leads PRO?', 'easycf7-leads' ); ?></strong></p>
            <ul>
                <li><?php esc_html_e( 'Panel "Mis leads" para cada usuario.', 'easycf7-leads' ); ?></li>
                <li><?php esc_html_e( 'Asignación manual y masiva de leads a usuarios.', 'easycf7-leads' ); ?></li>
                <li><?php esc_html_e( 'Notificaciones por email al asignar leads.', 'easycf7-leads' ); ?></li>
                <li><?php esc_html_e( 'GeoIP y mapa aproximado del lead.', 'easycf7-leads' ); ?></li>
                <li><?php esc_html_e( 'Webhooks para integrar con otros sistemas.', 'easycf7-leads' ); ?></li>
            </ul>
        </div>
    </div>
    <?php
}


add_action( 'admin_init', 'ecf7l_process_export_csv' );

function ecf7l_process_export_csv() {

    if ( ! is_admin() ) {
        return;
    }

    if ( ! isset( $_GET['page'] ) || $_GET['page'] !== 'easycf7-leads' ) {
        return;
    }

    if ( ! isset( $_GET['ecf7l_export'] ) || $_GET['ecf7l_export'] !== 'csv' ) {
        return;
    }

    // Nonce
    if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'ecf7l_export_leads' ) ) {
        wp_die( 'Nonce inválido.' );
    }

    // Cargar clase
    if ( ! class_exists( 'ECF7L_Leads' ) ) {
        return;
    }

    // Recoger filtros igual que en la lista
    $args = array(
        'paged'           => 1,
        'per_page'        => 999999,
        'search'          => isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '',
        'status'          => isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '',
        'form_id'         => isset( $_GET['form_id'] ) ? (int) $_GET['form_id'] : '',
        'date_from'       => isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '',
        'date_to'         => isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '',
        'assigned_to'     => isset( $_GET['assigned_to'] ) ? sanitize_text_field( wp_unslash( $_GET['assigned_to'] ) ) : '',
        'unassigned_only' => isset( $_GET['unassigned_only'] ) ? (int) $_GET['unassigned_only'] : 0,
        'orderby'         => isset( $_GET['orderby'] ) ? sanitize_text_field( wp_unslash( $_GET['orderby'] ) ) : 'date',
        'order'           => isset( $_GET['order'] ) ? sanitize_text_field( wp_unslash( $_GET['order'] ) ) : 'desc',
    );

    // Obtener datos
    $results = ECF7L_Leads::get_all( $args );
    $items   = $results['items'];

    // Cabeceras CSV
    nocache_headers();
    header( 'Content-Type: text/csv; charset=utf-8' );
    header( 'Content-Disposition: attachment; filename="easycf7-leads-' . date_i18n( 'Ymd-His' ) . '.csv"' );

    $output = fopen( 'php://output', 'w' );
    fprintf( $output, chr(0xEF) . chr(0xBB) . chr(0xBF) ); // UTF-8 BOM
    fputcsv( $output, array( 'ID', 'Formulario', 'Nombre', 'Email', 'Teléfono', 'Estado', 'Etiquetas', 'Responsable', 'Fecha' ), ';' );

    foreach ( $items as $lead ) {
        fputcsv(
            $output,
            array(
                $lead['id'],
                $lead['form_title'],
                $lead['name'],
                $lead['email'],
                $lead['phone'],
                $lead['status'],
                $lead['tags'],
                $lead['assigned_to'],
                $lead['submitted_at'],
            ),
            ';'
        );
    }

    fclose( $output );
    exit;
}

// Exportar leads a CSV vía AJAX (respeta filtros actuales de la lista).
add_action( 'wp_ajax_ecf7l_export_leads', 'ecf7l_export_leads_ajax' );

function ecf7l_export_leads_ajax() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( 'No tienes permisos para exportar leads.', 'easycf7-leads' ) );
    }

    if ( ! class_exists( 'ECF7L_Leads' ) ) {
        wp_die( 'ECF7L_Leads no está disponible.' );
    }

    $is_pro = ecf7l_is_pro();

    // Filtros igual que en la lista
    $search  = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
    $status  = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
    $form_id = isset( $_GET['form_id'] ) ? (int) $_GET['form_id'] : 0;

    $date_from = isset( $_GET['date_from'] ) ? sanitize_text_field( wp_unslash( $_GET['date_from'] ) ) : '';
    $date_to   = isset( $_GET['date_to'] ) ? sanitize_text_field( wp_unslash( $_GET['date_to'] ) ) : '';

    $assigned_to_raw     = isset( $_GET['assigned_to'] ) ? wp_unslash( $_GET['assigned_to'] ) : '';
    $assigned_to         = ( '' !== $assigned_to_raw ) ? $assigned_to_raw : '';
    $unassigned_only     = isset( $_GET['unassigned_only'] ) && '1' === $_GET['unassigned_only'];
    $unassigned_only_eff = $is_pro ? $unassigned_only : false;

    $orderby = isset( $_GET['orderby'] ) ? sanitize_text_field( wp_unslash( $_GET['orderby'] ) ) : 'date';
    $order   = isset( $_GET['order'] ) ? strtolower( sanitize_text_field( wp_unslash( $_GET['order'] ) ) ) : 'desc';
    $order   = ( 'asc' === $order ) ? 'asc' : 'desc';

    $args = array(
        'paged'           => 1,
        'per_page'        => 999999,
        'search'          => $search,
        'status'          => $status,
        'form_id'         => $form_id,
        'date_from'       => $date_from,
        'date_to'         => $date_to,
        'assigned_to'     => $assigned_to,
        'unassigned_only' => $unassigned_only_eff ? 1 : 0,
        'orderby'         => $orderby,
        'order'           => $order,
    );

    $results  = ECF7L_Leads::get_all( $args );
    $items    = $results['items'];
    $statuses = ECF7L_Leads::get_statuses();

    // Mapa de usuarios para el nombre del responsable
    $users = get_users(
        array(
            'role__in' => array( 'administrator', 'editor', 'author' ),
            'orderby'  => 'display_name',
            'order'    => 'ASC',
        )
    );
    $user_map = array();
    foreach ( $users as $u ) {
        $user_map[ $u->ID ] = $u->display_name;
    }

    if ( ! headers_sent() ) {
        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="easycf7-leads-' . date_i18n( 'Ymd-His' ) . '.csv"' );
    }

    $output = fopen( 'php://output', 'w' );

    // BOM UTF-8 para Excel
    fprintf( $output, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

    fputcsv(
        $output,
        array( 'ID', 'Formulario', 'Nombre', 'Email', 'Teléfono', 'Estado', 'Etiquetas', 'Responsable', 'Fecha' ),
        ';'
    );

    foreach ( $items as $lead ) {
        $lead_id       = (int) $lead['id'];
        $status_key    = $lead['status'] ? $lead['status'] : 'nuevo';
        $status_label  = isset( $statuses[ $status_key ] ) ? $statuses[ $status_key ] : $status_key;
        $tags_str      = trim( (string) $lead['tags'] );
        $assigned_id   = (int) $lead['assigned_to'];
        $assigned_name = $assigned_id && isset( $user_map[ $assigned_id ] ) ? $user_map[ $assigned_id ] : '';

        fputcsv(
            $output,
            array(
                $lead_id,
                $lead['form_title'],
                $lead['name'],
                $lead['email'],
                $lead['phone'],
                $status_label,
                $tags_str,
                $assigned_name,
                $lead['submitted_at'],
            ),
            ';'
        );
    }

    fclose( $output );
    exit;
}


add_filter( 'plugin_row_meta', 'easycf7leads_plugin_row_meta', 10, 2 );
function easycf7leads_plugin_row_meta( $links ) {
        $new_links = [
            '<a href="https://huwawa.shop" target="_blank">Visita la web del plugin</a>',
        ];
        $links = array_merge( $links, $new_links );
    return $links;
}